/* 
 * @(#)Fraktal.java Fraktalprogramm Mandelbrotmenge 
 *                  Fractal Program Mandelbrot Set 
 * © Copyright Eckhard Roessel. All Rights Reserved.
 * 30.06.2000 - 16.01.2010 - 07.02.2014
 */

import java.applet.Applet;
import java.awt.Image;
import java.awt.Graphics;
import java.awt.Cursor;
import java.awt.Color;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseEvent;

/**
 * @version 1.2
 * @author Eckhard Roessel
 * @modified 07.02.2014
 */
public class Fraktal extends Applet implements MouseListener, MouseMotionListener {

    private final static int MAX = 256;       // Anzahl maximale Iterationen - maximum iterations
    private final static double SX = -2.025;  // Startwert Realteil          - start value real
    private final static double SY = -1.125;  // Startwert Imaginärteil      - start value imaginary
    private final static double EX = 0.6;     // Endwert Realteil            - end value real
    private final static double EY = 1.125;   // Endwert Imaginärteil        - end value imaginary
    private int x1, y1, xs, ys, xe, ye;
    private double xstart, ystart, xende, yende, xzoom, yzoom;
    private boolean action, rechteck, fertig;
    private float xy, klein;
    private Image bild;
    private Graphics g1;
    private Cursor c1, c2;

    public void init() // alle Instanzen bereitstellen - all instances will be prepared
    {
        fertig = false;
        addMouseListener(this);
        addMouseMotionListener(this);
        c1 = new Cursor(Cursor.WAIT_CURSOR);
        c2 = new Cursor(Cursor.CROSSHAIR_CURSOR);
        x1 = getSize().width;       // Bildweite
        y1 = getSize().height;      // Bildhöhe
        xy = (float) x1 / (float) y1;
        bild = createImage(x1, y1); // Bildspeicher erzeugen
        g1 = bild.getGraphics();
        klein = (float) 1.0e-7;
        fertig = true;
        start();
        repaint();
    }

    public void destroy() // alle Instanzen freigeben - delete all instances 
    {
        if (fertig) {
            removeMouseListener(this);
            removeMouseMotionListener(this);
            bild = null;
            g1 = null;
            c1 = null;
            c2 = null;
        }
    }

    public void start() {
        action = false;
        rechteck = false;
        startwerte();
        xzoom = (xende - xstart) / (double) x1;
        yzoom = (yende - ystart) / (double) y1;
        mandelbrot();
    }

    public void stop() {
    }

    public void paint(Graphics g) {
        update(g);
    }

    public void update(Graphics g) {
        g.drawImage(bild, 0, 0, this);
        if (rechteck) {
            g.setColor(Color.white);
            if (xs < xe) {
                if (ys < ye) {
                    g.drawRect(xs, ys, (xe - xs), (ye - ys));
                } else {
                    g.drawRect(xs, ye, (xe - xs), (ys - ye));
                }
            } else {
                if (ys < ye) {
                    g.drawRect(xe, ys, (xs - xe), (ye - ys));
                } else {
                    g.drawRect(xe, ye, (xs - xe), (ys - ye));
                }
            }
        }
    }

    private void mandelbrot() // Alle Punkte berechnen - calculate all points
    {
        int x, y;
        float h, b, alt;
        alt = 0.0f;
        action = false;
        setCursor(c1);
        showStatus("bitte warten... - please wait...");
        for (x = 0; x < x1; x += 2) {
            for (y = 0; y < y1; y++) {
                h = punktfarbe(xstart + xzoom * (double) x, ystart + yzoom * (double) y);
                if (Math.abs(h - alt) > klein) {
                    b = 1.0f - h * h; // Helligkeit -  brightness
                    g1.setColor(Color.getHSBColor(h, 0.8f, b));
                    alt = h;
                }
                g1.drawLine(x, y, x + 1, y);
            }
        }
        showStatus("Mandelbrotmenge fertig - Mandelbrot Set ready");
        setCursor(c2);
        action = true;
    }

    private float punktfarbe(double xwert, double ywert) // Farbwert 0.0 bis 1.0 nach Anzahl der Iterationen
    {                                                    // color value from 0.0 to 1.0 by iterations
        double r = 0.0, i = 0.0, m = 0.0;
        int j = 0;

        while ((j < MAX) && (m < 4.0)) {
            j++;
            m = r * r - i * i;
            i = 2.0 * r * i + ywert;
            r = m + xwert;
        }
        return (float) j / (float) MAX;
    }

    private void startwerte() // Zurücksetzen auf Startwerte - reset to start values
    {
        xstart = SX;
        ystart = SY;
        xende = EX;
        yende = EY;
        if (Math.abs((float) ((xende - xstart) / (yende - ystart)) - xy) > klein) {
            xstart = xende - (yende - ystart) * (double) xy;
        }
    }

    public void mousePressed(MouseEvent e) // Maus gedrückt
    {
        e.consume();
        if (action) {
            xs = e.getX(); // Mausposition Startpunkt - start point
            ys = e.getY();
        }
    }

    public void mouseReleased(MouseEvent e) // Maus losgelassen - end point
    {
        int z, w;

        e.consume();
        if (action) {
            xe = e.getX();
            ye = e.getY();
            if (xs > xe) {
                z = xs;
                xs = xe;
                xe = z;
            }
            if (ys > ye) {
                z = ys;
                ys = ye;
                ye = z;
            }
            w = (xe - xs);
            z = (ye - ys);
            if ((w < 2) && (z < 2)) {
                startwerte();
            } else {
                if (((float) w > (float) z * xy)) {
                    ye = (int) ((float) ys + (float) w / xy);
                } else {
                    xe = (int) ((float) xs + (float) z * xy);
                }
                xende = xstart + xzoom * (double) xe;
                yende = ystart + yzoom * (double) ye;
                xstart += xzoom * (double) xs;
                ystart += yzoom * (double) ys;
            }
            xzoom = (xende - xstart) / (double) x1;
            yzoom = (yende - ystart) / (double) y1;
            mandelbrot();
            rechteck = false;
            repaint();
        }
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void mouseClicked(MouseEvent e) {
    }

    public void mouseDragged(MouseEvent e) {
        e.consume();
        if (action) {
            xe = e.getX();
            ye = e.getY();
            rechteck = true;
            repaint();
        }
    }

    public void mouseMoved(MouseEvent e) {
    }
}